<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

/**
 * Service para PIX Manual
 * Migrado do sistema legado: /legado/master/api/gerar_pix_link.php
 * 
 * PIX Manual não requer integração com API externa.
 * Apenas retorna a chave PIX cadastrada pelo usuário.
 */
class PixManualService
{
    private string $chavePix;

    public function __construct(string $chavePix)
    {
        $this->chavePix = $chavePix;
    }

    /**
     * Gera pagamento PIX Manual
     * Retorna apenas a chave PIX para o cliente copiar e colar
     */
    public function generatePayment(array $options): array
    {
        try {
            if (empty($this->chavePix)) {
                Log::error('PIX Manual: Chave PIX não configurada');
                return [
                    'error' => true,
                    'message' => 'Chave PIX não configurada. Entre em contato com o suporte.'
                ];
            }

            $idPayment = 'PIX_MANUAL_' . time() . '_' . uniqid();
            $statusPayment = 'pending';

            return [
                'id_payment' => $idPayment,
                'status_payment' => $statusPayment,
                'copiacola' => $this->chavePix,
                'imagemQrcode' => '', // PIX Manual não gera QR Code automaticamente
                'codigobarra' => '',
                'createdDate' => Carbon::now()->format('Y-m-d H:i:s'),
                'valorparcela' => $options['item_valor'] ?? 0,
            ];

        } catch (\Exception $e) {
            Log::error('PIX Manual: Exceção ao gerar pagamento', [
                'error' => $e->getMessage()
            ]);
            return [
                'error' => true,
                'message' => 'Erro ao processar PIX Manual: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Gera QR Code estático a partir da chave PIX (opcional)
     * Requer biblioteca chillerlan/php-qrcode
     */
    public function generateQRCode(?string $valor = null, ?string $descricao = null): ?string
    {
        try {
            // Verifica se a biblioteca está disponível
            if (!class_exists(\chillerlan\QRCode\QRCode::class)) {
                Log::warning('PIX Manual: Biblioteca chillerlan/php-qrcode não disponível');
                return null;
            }

            // Monta payload PIX (EMV)
            $payload = $this->buildPixPayload($valor, $descricao);

            // Gera QR Code
            $qrcode = new \chillerlan\QRCode\QRCode();
            $qrcodeBase64 = $qrcode->render($payload);

            // Remove prefixo data:image se existir
            if (strpos($qrcodeBase64, 'data:image') !== false) {
                $qrcodeBase64 = preg_replace('/^data:image\/[^;]+;base64,/', '', $qrcodeBase64);
            }

            return $qrcodeBase64;

        } catch (\Exception $e) {
            Log::error('PIX Manual: Erro ao gerar QR Code', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Monta payload PIX (EMV) para QR Code estático
     */
    private function buildPixPayload(?string $valor = null, ?string $descricao = null): string
    {
        $payload = '00020126'; // Payload Format Indicator + Merchant Account Information
        
        // Merchant Account Information
        $merchantAccount = '0014BR.GOV.BCB.PIX01' . strlen($this->chavePix) . $this->chavePix;
        $payload .= '26' . str_pad(strlen($merchantAccount), 2, '0', STR_PAD_LEFT) . $merchantAccount;
        
        // Merchant Category Code (opcional)
        $payload .= '52040000'; // MCC genérico
        
        // Transaction Currency (BRL = 986)
        $payload .= '5303986';
        
        // Transaction Amount (opcional)
        if ($valor) {
            $valorFormatado = number_format((float) $valor, 2, '.', '');
            $payload .= '54' . str_pad(strlen($valorFormatado), 2, '0', STR_PAD_LEFT) . $valorFormatado;
        }
        
        // Country Code (BR = 076)
        $payload .= '5802BR';
        
        // Merchant Name (opcional)
        if ($descricao) {
            $payload .= '59' . str_pad(strlen($descricao), 2, '0', STR_PAD_LEFT) . $descricao;
        }
        
        // Additional Data Field Template (opcional)
        // $payload .= '62' . str_pad(strlen($additionalData), 2, '0', STR_PAD_LEFT) . $additionalData;
        
        // CRC16
        $crc = $this->crc16($payload . '6304');
        $payload .= '6304' . str_pad(dechex($crc), 4, '0', STR_PAD_LEFT);
        
        return $payload;
    }

    /**
     * Calcula CRC16 para PIX
     */
    private function crc16(string $str): int
    {
        $crc = 0xFFFF;
        $strlen = strlen($str);
        
        for ($c = 0; $c < $strlen; $c++) {
            $crc ^= ord($str[$c]) << 8;
            for ($i = 0; $i < 8; $i++) {
                if ($crc & 0x8000) {
                    $crc = ($crc << 1) ^ 0x1021;
                } else {
                    $crc = $crc << 1;
                }
            }
        }
        
        return $crc & 0xFFFF;
    }

    /**
     * Valida chave PIX
     */
    public function validateChavePix(): bool
    {
        if (empty($this->chavePix)) {
            return false;
        }

        // Valida formato de chave PIX
        // Pode ser: CPF, CNPJ, Email, Telefone, ou Chave Aleatória (UUID)
        $patterns = [
            '/^[0-9]{11}$/', // CPF
            '/^[0-9]{14}$/', // CNPJ
            '/^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$/', // Email
            '/^\+[0-9]{10,13}$/', // Telefone
            '/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', // UUID
        ];

        foreach ($patterns as $pattern) {
            if (preg_match($pattern, $this->chavePix)) {
                return true;
            }
        }

        // Se não corresponde a nenhum padrão, pode ser chave aleatória válida
        return strlen($this->chavePix) >= 10;
    }
}

