<?php

namespace App\Services;

use App\Models\PdvCaixa;
use App\Models\PdvVenda;
use App\Models\PdvProduto;
use App\Models\PdvItem;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PdvService
{
    /**
     * Abre um novo caixa
     */
    public function abrirCaixa(int $userId, float $saldoInicial, ?string $atendente = null): PdvCaixa
    {
        return DB::transaction(function () use ($userId, $saldoInicial, $atendente) {
            // Verificar se o usuário existe (o sistema usa Carteira, não User)
            try {
                $user = \App\Models\Carteira::findOrFail($userId);
            } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
                Log::error('Usuário não encontrado ao abrir caixa', [
                    'user_id' => $userId,
                ]);
                throw new \Exception('Usuário não encontrado! Verifique se você está logado corretamente.');
            }

            // Verificar se já existe caixa aberto (verificar também por NULL ou qualquer status diferente de 'fechado')
            $caixaAberto = PdvCaixa::where('idm', $userId)
                ->where(function($query) {
                    $query->where('status', 'aberto')
                          ->orWhereNull('status')
                          ->orWhere('status', '!=', 'fechado');
                })
                ->first();

            if ($caixaAberto) {
                // Se o status não está definido como 'fechado', considerar como aberto
                if ($caixaAberto->status !== 'fechado') {
                    throw new \Exception('Já existe um caixa aberto (ID: ' . $caixaAberto->id . ')! Feche o caixa atual antes de abrir um novo.');
                }
            }

            try {
                $caixa = PdvCaixa::create([
                    'idm' => $userId,
                    'abertura' => now(),
                    'saldo_inicial' => $saldoInicial,
                    'atendente_nome' => $atendente ?: null,
                    'status' => 'aberto',
                ]);

                Log::info('Caixa aberto', [
                    'caixa_id' => $caixa->id,
                    'user_id' => $userId,
                    'saldo_inicial' => $saldoInicial,
                ]);

                return $caixa;
            } catch (\Illuminate\Database\QueryException $e) {
                Log::error('Erro ao abrir caixa', [
                    'user_id' => $userId,
                    'error' => $e->getMessage(),
                    'code' => $e->getCode(),
                ]);

                // Verificar se é erro de foreign key
                if ($e->getCode() == 23000) {
                    if (strpos($e->getMessage(), 'foreign key') !== false) {
                        throw new \Exception('Erro: Usuário não encontrado ou inválido.');
                    } elseif (strpos($e->getMessage(), 'Duplicate entry') !== false) {
                        throw new \Exception('Erro: Já existe um caixa com essas informações.');
                    }
                }

                throw new \Exception('Erro ao abrir caixa: ' . $e->getMessage());
            }
        });
    }

    /**
     * Fecha um caixa
     */
    public function fecharCaixa(PdvCaixa $caixa): PdvCaixa
    {
        return DB::transaction(function () use ($caixa) {
            if (!$caixa->isAberto()) {
                throw new \Exception('Caixa já está fechado!');
            }

            $totalVendas = $caixa->vendas()
                ->where('status', 'ativa')
                ->sum('total');

            $saldoFinal = $caixa->saldo_inicial + $totalVendas;

            $caixa->update([
                'fechamento' => now(),
                'saldo_final' => $saldoFinal,
                'status' => 'fechado',
            ]);

            Log::info('Caixa fechado', [
                'caixa_id' => $caixa->id,
                'saldo_final' => $saldoFinal,
                'total_vendas' => $totalVendas,
            ]);

            return $caixa;
        });
    }

    /**
     * Cria uma nova venda
     */
    public function criarVenda(array $dados): PdvVenda
    {
        return DB::transaction(function () use ($dados) {
            $venda = PdvVenda::create([
                'idm' => $dados['idm'],
                'caixa_id' => $dados['caixa_id'],
                'cliente_id' => $dados['cliente_id'] ?? null,
                'funcionario_id' => $dados['funcionario_id'] ?? null,
                'total' => $dados['total'],
                'desconto' => $dados['desconto'] ?? 0,
                'forma_pagamento' => $dados['forma_pagamento'],
                'data_venda' => now(),
                'status' => 'ativa',
            ]);

            // Adicionar itens
            foreach ($dados['itens'] as $item) {
                $produto = PdvProduto::findOrFail($item['produto_id']);

                // Verificar estoque
                if ($produto->estoque < $item['quantidade']) {
                    throw new \Exception("Estoque insuficiente para o produto: {$produto->nome}");
                }

                PdvItem::create([
                    'id_venda' => $venda->id,
                    'id_produto' => $item['produto_id'],
                    'quantidade' => $item['quantidade'],
                    'preco_unitario' => $item['preco'],
                    'subtotal' => $item['quantidade'] * $item['preco'],
                ]);

                // Atualizar estoque
                $produto->removerEstoque($item['quantidade']);
            }

            // Calcular comissão se houver funcionário
            if ($venda->funcionario_id) {
                $funcionario = Funcionario::find($venda->funcionario_id);
                if ($funcionario && $funcionario->porcentagem_vendas > 0) {
                    $comissao = ($venda->total * $funcionario->porcentagem_vendas) / 100;
                    $venda->comissao = $comissao;
                    $venda->save();
                }
            }

            Log::info('Venda criada', [
                'venda_id' => $venda->id,
                'total' => $venda->total,
                'itens' => count($dados['itens']),
            ]);

            return $venda;
        });
    }

    /**
     * Cancela uma venda
     */
    public function cancelarVenda(PdvVenda $venda): PdvVenda
    {
        return DB::transaction(function () use ($venda) {
            if (!$venda->isAtiva()) {
                throw new \Exception('Venda já está cancelada ou trocada!');
            }

            // Devolver estoque
            foreach ($venda->itens as $item) {
                $produto = $item->produto;
                $produto->adicionarEstoque($item->quantidade);
            }

            $venda->update([
                'status' => 'cancelada',
            ]);

            Log::info('Venda cancelada', [
                'venda_id' => $venda->id,
            ]);

            return $venda;
        });
    }
}

