<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

/**
 * Service para integração com Mercado Pago
 * Migrado do sistema legado: /legado/mercadopago/gerarpagamentos.php
 */
class MercadoPagoService
{
    private string $accessToken;
    private string $notificationUrl;

    public function __construct(string $accessToken, ?string $notificationUrl = null)
    {
        $this->accessToken = $accessToken;
        $this->notificationUrl = $notificationUrl ?? url('/webhook/mercadopago');
    }

    /**
     * Gera pagamento PIX ou Boleto
     */
    public function createPayment(array $options, string $type = 'pix'): ?array
    {
        try {
            // Validação de CPF/CNPJ para boleto
            if ($type === 'boleto') {
                $cpf_cnpj = str_replace(['.', '-', '/'], '', $options['cpf'] ?? '');
                $cpf_cnpj_type = strlen($cpf_cnpj) > 11 ? 'CNPJ' : 'CPF';
                
                if ($cpf_cnpj_type === 'CPF' && strlen($cpf_cnpj) != 11) {
                    return ['error' => true, 'message' => 'CPF inválido.'];
                }
                
                if ($cpf_cnpj_type === 'CNPJ' && strlen($cpf_cnpj) != 14) {
                    return ['error' => true, 'message' => 'CNPJ inválido.'];
                }
            }

            // Processa data de vencimento
            $vencimento = $this->processDueDate($options['vencimento'] ?? null);
            $dateOfExpiration = $vencimento
                ->setTime(23, 59, 59)
                ->setTimezone(new \DateTimeZone('America/Sao_Paulo'))
                ->format('Y-m-d\TH:i:s.vP');

            // Monta dados do pagador
            $payer = $this->buildPayer($options, $type);

            // Monta preferência
            $preference = [
                'transaction_amount' => (float) ($options['item_valor'] ?? 0),
                'description' => $options['item_description'] ?? 'PAGAMENTO DE MENSALIDADE',
                'notification_url' => $this->notificationUrl,
                'payer' => $payer,
                'payment_method_id' => $type === 'boleto' ? 'bolbradesco' : 'pix',
                'date_of_expiration' => $dateOfExpiration,
            ];

            // Faz requisição
            $response = $this->request('v1/payments', $preference, 'POST');

            if (!$response || isset($response['error'])) {
                Log::error('Mercado Pago: Erro ao criar pagamento', [
                    'response' => $response,
                    'options' => $options
                ]);
                return null;
            }

            // Processa resposta
            return $this->processPaymentResponse($response, $type);

        } catch (\Exception $e) {
            Log::error('Mercado Pago: Exceção ao criar pagamento', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    /**
     * Processa data de vencimento
     */
    private function processDueDate(?string $vencimento): Carbon
    {
        $dataRequisicao = Carbon::now();
        
        if ($vencimento) {
            try {
                $dataVencimento = Carbon::createFromFormat('d/m/Y', $vencimento);
            } catch (\Exception $e) {
                $dataVencimento = $dataRequisicao->copy()->addDays(3);
            }
        } else {
            $dataVencimento = $dataRequisicao->copy()->addDays(3);
        }

        // Se vencimento é menor que hoje, adiciona 3 dias
        if ($dataVencimento->lt($dataRequisicao)) {
            $dataVencimento = $dataRequisicao->copy()->addDays(3);
        }

        // Máximo de 28 dias
        $diferenca = $dataRequisicao->diffInDays($dataVencimento);
        if ($diferenca > 28) {
            $dataVencimento = $dataRequisicao->copy()->addDays(28);
        }

        return $dataVencimento;
    }

    /**
     * Monta dados do pagador
     */
    private function buildPayer(array $options, string $type): array
    {
        if ($type === 'boleto') {
            $nameParts = explode(' ', $options['name'] ?? '');
            $firstName = $nameParts[0] ?? '';
            $lastName = end($nameParts) ?? '';
            $cpf_cnpj = str_replace(['.', '-', '/'], '', $options['cpf'] ?? '');
            $cpf_cnpj_type = strlen($cpf_cnpj) > 11 ? 'CNPJ' : 'CPF';

            $clienteDb = $options['cliente_db'] ?? null;

            return [
                'email' => $options['email'] ?? '',
                'first_name' => $firstName,
                'last_name' => $lastName,
                'identification' => [
                    'type' => $cpf_cnpj_type,
                    'number' => $cpf_cnpj
                ],
                'address' => [
                    'zip_code' => $clienteDb->cep ?? '00000000',
                    'street_name' => $clienteDb->endereco ?? 'Rua Teste',
                    'street_number' => $clienteDb->numero ?? '123',
                    'neighborhood' => $clienteDb->bairro ?? 'Bairro Teste',
                    'city' => $clienteDb->cidade ?? 'Cidade Teste',
                    'federal_unit' => $clienteDb->uf ?? 'EX',
                ]
            ];
        }

        return [
            'email' => $options['email'] ?? ''
        ];
    }

    /**
     * Processa resposta do pagamento
     */
    private function processPaymentResponse(array $response, string $type): array
    {
        $result = [
            'id_payment' => $response['id'] ?? '',
            'status_payment' => $response['status'] ?? '',
            'valorparcela' => $response['transaction_details']['total_paid_amount'] ?? null,
            'createdDate' => Carbon::now()->format('Y-m-d H:i:s'),
        ];

        if ($type === 'pix') {
            $result['imagemQrcode'] = $response['point_of_interaction']['transaction_data']['qr_code_base64'] ?? '';
            $result['copiacola'] = $response['point_of_interaction']['transaction_data']['qr_code'] ?? '';
        } else {
            $result['codigobarra'] = $response['transaction_details']['digitable_line'] ?? '';
            $result['ticket_url'] = $response['point_of_interaction']['transaction_data']['ticket_url'] ?? '';
        }

        return $result;
    }

    /**
     * Consulta status de pagamento
     */
    public function getPaymentStatus(string $paymentId): ?string
    {
        try {
            $response = $this->request("v1/payments/{$paymentId}", [], 'GET');

            if (!$response || isset($response['error'])) {
                Log::error('Mercado Pago: Erro ao consultar pagamento', [
                    'payment_id' => $paymentId,
                    'response' => $response
                ]);
                return null;
            }

            return $response['status'] ?? null;

        } catch (\Exception $e) {
            Log::error('Mercado Pago: Exceção ao consultar pagamento', [
                'payment_id' => $paymentId,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Busca detalhes do pagamento
     */
    public function getPayment(string $paymentId): ?array
    {
        try {
            $response = $this->request("v1/payments/{$paymentId}", [], 'GET');

            if (!$response || isset($response['error'])) {
                return null;
            }

            return [
                'id' => $response['id'] ?? null,
                'status' => $response['status'] ?? null,
                'status_detail' => $response['status_detail'] ?? null,
                'transaction_amount' => $response['transaction_amount'] ?? null,
                'date_approved' => $response['date_approved'] ?? null,
                'date_created' => $response['date_created'] ?? null,
                'payment_method_id' => $response['payment_method_id'] ?? null,
                'point_of_interaction' => $response['point_of_interaction'] ?? null,
            ];

        } catch (\Exception $e) {
            Log::error('Mercado Pago: Exceção ao buscar pagamento', [
                'payment_id' => $paymentId,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Cria preferência de pagamento (Checkout Transparente)
     */
    public function createPreference(array $items, array $payer, ?string $externalReference = null): ?string
    {
        try {
            $preference = [
                'items' => $items,
                'payer' => $payer,
                'notification_url' => $this->notificationUrl,
            ];

            if ($externalReference) {
                $preference['external_reference'] = $externalReference;
            }

            $response = $this->request('checkout/preferences', $preference, 'POST');

            if (!$response || isset($response['error'])) {
                Log::error('Mercado Pago: Erro ao criar preferência', [
                    'response' => $response
                ]);
                return null;
            }

            return $response['id'] ?? null;

        } catch (\Exception $e) {
            Log::error('Mercado Pago: Exceção ao criar preferência', [
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Faz requisição HTTP para API Mercado Pago
     */
    private function request(string $endpoint, array $data = [], string $method = 'POST'): ?array
    {
        try {
            $url = 'https://api.mercadopago.com/' . ltrim($endpoint, '/');

            $headers = [
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->accessToken,
            ];

            if ($method === 'POST') {
                $headers['X-Idempotency-Key'] = uniqid();
            }

            $request = Http::withHeaders($headers);

            $response = match($method) {
                'GET' => $request->get($url),
                'POST' => $request->post($url, $data),
                'PUT' => $request->put($url, $data),
                'DELETE' => $request->delete($url),
                default => null
            };

            if ($response && $response->successful()) {
                return $response->json();
            }

            Log::error('Mercado Pago: Erro na requisição', [
                'endpoint' => $endpoint,
                'method' => $method,
                'status' => $response?->status(),
                'body' => $response?->body()
            ]);

            return null;

        } catch (\Exception $e) {
            Log::error('Mercado Pago: Exceção na requisição', [
                'endpoint' => $endpoint,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }
}

