<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class LicenseService
{
    private const LICENSE_API_URL = 'https://licencasgestor.plwstoremultishop.com/painel/controle/api/';
    private const CACHE_KEY = 'license_validation_';
    private const CACHE_DURATION = 3600; // 1 hora

    /**
     * Valida a licença do domínio
     */
    public function validateLicense(?string $domain = null): array
    {
        if (!$domain) {
            $domain = request()->getHost();
        }

        // Verifica cache primeiro
        $cacheKey = self::CACHE_KEY . md5($domain);
        $cached = Cache::get($cacheKey);
        
        if ($cached !== null) {
            return $cached;
        }

        try {
            $response = Http::timeout(10)
                ->withoutVerifying() // Ignora verificação SSL (como no legado)
                ->post(self::LICENSE_API_URL, [
                    'domain' => $domain
                ]);

            if ($response->successful()) {
                $data = $response->json();
                
                $result = [
                    'valid' => $data['valido'] ?? false,
                    'response' => $data['response'] ?? false,
                    'version' => $data['version'] ?? null,
                    'expiration_date' => $data['dataexp'] ?? null,
                    'message' => $data['message'] ?? null,
                ];

                // Cache o resultado
                Cache::put($cacheKey, $result, self::CACHE_DURATION);
                
                return $result;
            }

            Log::warning('License API returned non-successful response', [
                'status' => $response->status(),
                'domain' => $domain
            ]);

            return $this->getFailureResponse();

        } catch (\Exception $e) {
            Log::error('License validation failed', [
                'domain' => $domain,
                'error' => $e->getMessage()
            ]);

            return $this->getFailureResponse();
        }
    }

    /**
     * Verifica se a licença está válida
     */
    public function isValid(?string $domain = null): bool
    {
        $validation = $this->validateLicense($domain);
        return $validation['valid'] && $validation['response'];
    }

    /**
     * Obtém a versão atual do sistema remoto
     */
    public function getRemoteVersion(?string $domain = null): ?string
    {
        $validation = $this->validateLicense($domain);
        return $validation['version'] ?? null;
    }

    /**
     * Obtém a data de expiração da licença
     */
    public function getExpirationDate(?string $domain = null): ?string
    {
        $validation = $this->validateLicense($domain);
        return $validation['expiration_date'] ?? null;
    }

    /**
     * Limpa o cache de validação
     */
    public function clearCache(?string $domain = null): void
    {
        if (!$domain) {
            $domain = request()->getHost();
        }
        
        $cacheKey = self::CACHE_KEY . md5($domain);
        Cache::forget($cacheKey);
    }

    /**
     * Resposta padrão para falhas
     */
    private function getFailureResponse(): array
    {
        return [
            'valid' => false,
            'response' => false,
            'version' => null,
            'expiration_date' => null,
            'message' => 'Não foi possível validar a licença'
        ];
    }

    /**
     * Verifica se há nova versão disponível
     */
    public function hasNewVersion(): bool
    {
        $remoteVersion = $this->getRemoteVersion();
        $currentVersion = config('app.version', '1.1.0');

        if (!$remoteVersion) {
            return false;
        }

        return version_compare($remoteVersion, $currentVersion, '>');
    }
}

