<?php

namespace App\Services;

use App\Models\Financeiro2;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class JurosService
{
    /**
     * Calcula e aplica juros em parcelas vencidas
     */
    public function calcularJurosVencidos($carteiraId = null)
    {
        $query = Financeiro2::where('status', '1'); // Apenas parcelas abertas

        if ($carteiraId) {
            $query->where('idm', $carteiraId);
        }

        $parcelas = $query->get();
        $parcelasAtualizadas = 0;

        foreach ($parcelas as $parcela) {
            try {
                if ($this->aplicarJuros($parcela)) {
                    $parcelasAtualizadas++;
                }
            } catch (\Exception $e) {
                Log::error("Erro ao calcular juros da parcela {$parcela->id}: " . $e->getMessage());
            }
        }

        return [
            'total_parcelas' => $parcelas->count(),
            'parcelas_atualizadas' => $parcelasAtualizadas,
        ];
    }

    /**
     * Aplica juros em uma parcela específica
     */
    public function aplicarJuros(Financeiro2 $parcela)
    {
        // Verifica se a parcela está vencida
        if (!$parcela->isVencido()) {
            return false;
        }

        // Busca taxa de juros da carteira
        $carteira = $parcela->carteira;
        if (!$carteira || $carteira->juros_diarios <= 0) {
            return false;
        }

        // Calcula dias vencidos
        $diasVencidos = $parcela->calcularDiasVencidos();
        if ($diasVencidos <= 0) {
            return false;
        }

        // Calcula juros
        $valorOriginal = $parcela->parcela - ($parcela->taxa_juros_diaria * $parcela->dias_vencidos);
        $jurosAplicar = $valorOriginal * ($carteira->juros_diarios / 100) * $diasVencidos;

        // Atualiza parcela
        $parcela->update([
            'taxa_juros_diaria' => $carteira->juros_diarios,
            'dias_vencidos' => $diasVencidos,
            'juros_calculados' => 1,
            'parcela' => $valorOriginal + $jurosAplicar,
        ]);

        return true;
    }

    /**
     * Remove juros de uma parcela
     */
    public function removerJuros(Financeiro2 $parcela)
    {
        if ($parcela->juros_calculados == 0) {
            return false;
        }

        // Calcula valor original
        $valorOriginal = $parcela->parcela - ($parcela->taxa_juros_diaria * $parcela->dias_vencidos);

        $parcela->update([
            'parcela' => $valorOriginal,
            'taxa_juros_diaria' => 0.00,
            'dias_vencidos' => 0,
            'juros_calculados' => 0,
        ]);

        return true;
    }

    /**
     * Calcula juros para simulação
     */
    public function simularJuros($valor, $taxaDiaria, $diasVencidos)
    {
        $juros = $valor * ($taxaDiaria / 100) * $diasVencidos;
        $valorFinal = $valor + $juros;

        return [
            'valor_original' => $valor,
            'taxa_diaria' => $taxaDiaria,
            'dias_vencidos' => $diasVencidos,
            'valor_juros' => $juros,
            'valor_final' => $valorFinal,
        ];
    }
}

