<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

/**
 * Service para integração com Asaas
 * Migrado do sistema legado: /legado/Asaas/request.php e gerarpagamento.php
 */
class AsaasService
{
    private string $accessToken;
    private bool $sandbox;

    public function __construct(string $accessToken, bool $sandbox = false)
    {
        $this->accessToken = $accessToken;
        $this->sandbox = $sandbox;
    }

    /**
     * Faz requisição para API Asaas
     */
    private function request(string $endpoint, array $body = [], string $method = 'POST'): ?array
    {
        try {
            $baseUrl = $this->sandbox 
                ? 'https://sandbox.asaas.com/api/v3/'
                : 'https://api.asaas.com/v3/';

            $request = Http::withHeaders([
                'accept' => 'application/json',
                'access_token' => $this->accessToken,
                'content-type' => 'application/json'
            ]);

            $response = match($method) {
                'GET' => $request->get($baseUrl . $endpoint),
                'POST' => $request->post($baseUrl . $endpoint, $body),
                'PUT' => $request->put($baseUrl . $endpoint, $body),
                'DELETE' => $request->delete($baseUrl . $endpoint),
                default => null
            };

            if ($response && $response->successful()) {
                return $response->json();
            }

            Log::error('Asaas: Erro na requisição', [
                'endpoint' => $endpoint,
                'status' => $response?->status(),
                'body' => $response?->body()
            ]);

            return null;

        } catch (\Exception $e) {
            Log::error('Asaas: Exceção na requisição', [
                'endpoint' => $endpoint,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Cria cliente na Asaas
     */
    public function createClient(array $dados): ?string
    {
        $body = [
            'name' => $dados['name'] ?? '',
            'cpfCnpj' => $dados['cpfCnpj'] ?? '',
            'email' => $dados['email'] ?? null,
            'phone' => $dados['phone'] ?? null,
            'mobilePhone' => $dados['mobilePhone'] ?? null,
            'address' => $dados['address'] ?? null,
            'addressNumber' => $dados['addressNumber'] ?? null,
            'complement' => $dados['complement'] ?? null,
            'province' => $dados['province'] ?? null,
            'postalCode' => $dados['postalCode'] ?? null,
            'externalReference' => $dados['externalReference'] ?? null,
        ];

        // Remove campos nulos
        $body = array_filter($body, fn($value) => $value !== null);

        $response = $this->request('customers', $body);
        
        return $response['id'] ?? null;
    }

    /**
     * Busca cliente por ID
     */
    public function getClient(string $customerId): ?array
    {
        return $this->request('customers/' . $customerId, [], 'GET');
    }

    /**
     * Cria cobrança
     */
    public function createPayment(array $dados): ?array
    {
        $body = [
            'customer' => $dados['customer'] ?? '',
            'billingType' => $dados['billingType'] ?? 'PIX', // PIX, BOLETO, CREDIT_CARD
            'value' => $dados['value'] ?? 0,
            'dueDate' => $dados['dueDate'] ?? date('Y-m-d'),
            'description' => $dados['description'] ?? '',
            'externalReference' => $dados['externalReference'] ?? null,
            'installmentCount' => $dados['installmentCount'] ?? null,
            'installmentValue' => $dados['installmentValue'] ?? null,
        ];

        // Remove campos nulos
        $body = array_filter($body, fn($value) => $value !== null);

        return $this->request('payments', $body);
    }

    /**
     * Gera QR Code PIX
     */
    public function generatePixQrCode(string $paymentId): ?array
    {
        $response = $this->request('payments/' . $paymentId . '/pixQrCode', [], 'GET');
        
        if ($response) {
            return [
                'success' => $response['success'] ?? false,
                'encodedImage' => $response['encodedImage'] ?? null,
                'payload' => $response['payload'] ?? null,
                'expirationDate' => $response['expirationDate'] ?? null,
            ];
        }

        return null;
    }

    /**
     * Consulta status de pagamento
     */
    public function getPaymentStatus(string $paymentId): ?array
    {
        $response = $this->request('payments/' . $paymentId, [], 'GET');
        
        if ($response) {
            return [
                'id' => $response['id'] ?? null,
                'status' => $response['status'] ?? null, // PENDING, RECEIVED, CONFIRMED, OVERDUE, REFUNDED, RECEIVED_IN_CASH, REFUND_REQUESTED, CHARGEBACK_REQUESTED, CHARGEBACK_DISPUTE, AWAITING_CHARGEBACK_REVERSAL, DUNNING_REQUESTED, DUNNING_RECEIVED, AWAITING_RISK_ANALYSIS
                'value' => $response['value'] ?? null,
                'netValue' => $response['netValue'] ?? null,
                'pago' => in_array($response['status'] ?? '', ['RECEIVED', 'CONFIRMED']),
                'billingType' => $response['billingType'] ?? null,
                'confirmedDate' => $response['confirmedDate'] ?? null,
                'paymentDate' => $response['paymentDate'] ?? null,
            ];
        }

        return null;
    }

    /**
     * Lista pagamentos de um cliente
     */
    public function listPayments(string $customerId, ?string $status = null): ?array
    {
        $endpoint = 'payments?customer=' . $customerId;
        
        if ($status) {
            $endpoint .= '&status=' . $status;
        }

        return $this->request($endpoint, [], 'GET');
    }

    /**
     * Cancela pagamento
     */
    public function cancelPayment(string $paymentId): bool
    {
        $response = $this->request('payments/' . $paymentId, [], 'DELETE');
        return $response !== null;
    }

    /**
     * Restaura pagamento cancelado
     */
    public function restorePayment(string $paymentId): bool
    {
        $response = $this->request('payments/' . $paymentId . '/restore', [], 'POST');
        return $response !== null;
    }

    /**
     * Gera link de pagamento
     */
    public function generatePaymentLink(string $paymentId): ?string
    {
        $response = $this->request('payments/' . $paymentId . '/identificationField', [], 'GET');
        return $response['identificationField'] ?? null;
    }

    /**
     * Webhook - Processa notificação
     */
    public function processWebhook(array $data): ?array
    {
        // Extrai informações do webhook
        return [
            'event' => $data['event'] ?? null,
            'payment' => $data['payment'] ?? null,
        ];
    }
}

