<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class PdvVenda extends Model
{
    use HasFactory;

    protected $table = '_pdv_vendas';

    protected $fillable = [
        'idm',
        'caixa_id',
        'cliente_id',
        'funcionario_id',
        'total',
        'desconto',
        'forma_pagamento',
        'data_venda',
        'status',
        'comissao',
    ];

    protected $casts = [
        'total' => 'decimal:2',
        'desconto' => 'decimal:2',
        'comissao' => 'decimal:2',
        'data_venda' => 'datetime',
    ];

    /**
     * Relacionamento com caixa
     */
    public function caixa(): BelongsTo
    {
        return $this->belongsTo(PdvCaixa::class, 'caixa_id');
    }

    /**
     * Relacionamento com cliente
     */
    public function cliente(): BelongsTo
    {
        return $this->belongsTo(Cliente::class, 'cliente_id', 'Id');
    }

    /**
     * Relacionamento com funcionário
     */
    public function funcionario(): BelongsTo
    {
        return $this->belongsTo(Funcionario::class, 'funcionario_id');
    }

    /**
     * Relacionamento com itens
     */
    public function itens(): HasMany
    {
        return $this->hasMany(PdvItem::class, 'id_venda');
    }

    /**
     * Relacionamento com usuário (Carteira)
     */
    public function usuario(): BelongsTo
    {
        return $this->belongsTo(Carteira::class, 'idm', 'id');
    }

    /**
     * Scope para vendas ativas
     */
    public function scopeAtivas($query)
    {
        return $query->where('status', 'ativa');
    }

    /**
     * Scope para vendas canceladas
     */
    public function scopeCanceladas($query)
    {
        return $query->where('status', 'cancelada');
    }

    /**
     * Calcula comissão baseada na porcentagem do funcionário
     */
    public function calcularComissao(float $porcentagem): float
    {
        return ($this->total * $porcentagem) / 100;
    }

    /**
     * Verifica se a venda está ativa
     */
    public function isAtiva(): bool
    {
        return $this->status === 'ativa';
    }
}

