<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class PdvCaixa extends Model
{
    use HasFactory;

    protected $table = '_pdv_caixa';

    protected $fillable = [
        'idm',
        'abertura',
        'fechamento',
        'saldo_inicial',
        'saldo_final',
        'atendente_nome',
        'status',
    ];

    protected $casts = [
        'abertura' => 'datetime',
        'fechamento' => 'datetime',
        'saldo_inicial' => 'decimal:2',
        'saldo_final' => 'decimal:2',
    ];

    /**
     * Relacionamento com vendas
     */
    public function vendas(): HasMany
    {
        return $this->hasMany(PdvVenda::class, 'caixa_id');
    }

    /**
     * Relacionamento com usuário (Carteira)
     */
    public function usuario(): BelongsTo
    {
        return $this->belongsTo(Carteira::class, 'idm', 'id');
    }

    /**
     * Scope para caixas abertos
     */
    public function scopeAberto($query)
    {
        return $query->where('status', 'aberto');
    }

    /**
     * Scope para caixas fechados
     */
    public function scopeFechado($query)
    {
        return $query->where('status', 'fechado');
    }

    /**
     * Verifica se o caixa está aberto
     */
    public function isAberto(): bool
    {
        return $this->status === 'aberto';
    }

    /**
     * Calcula o total de vendas ativas
     */
    public function getTotalVendasAttribute(): float
    {
        return $this->vendas()
            ->where('status', 'ativa')
            ->sum('total');
    }

    /**
     * Calcula o saldo final (saldo inicial + vendas)
     */
    public function calcularSaldoFinal(): float
    {
        return $this->saldo_inicial + $this->total_vendas;
    }
}

