<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Mensagem extends Model
{
    use HasFactory;

    protected $table = 'mensagens';

    protected $fillable = [
        'idu',
        'msg',
        'tipo',
        'status',
        'hora',
    ];

    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // Relacionamentos
    public function carteira()
    {
        return $this->belongsTo(Carteira::class, 'idu', 'id');
    }
    
    // Constantes
    const TIPO_5_DIAS_ANTES = '1';
    const TIPO_3_DIAS_ANTES = '2';
    const TIPO_VENCE_HOJE = '3';
    const TIPO_VENCIDO = '4';
    const TIPO_RECIBO = '5';
    const TIPO_PERSONALIZADO = '6';
    const TIPO_7_DIAS_ANTES = '7';
    
    // Métodos auxiliares
    public function isAtivo(): bool
    {
        return $this->status == '1';
    }
    
    public function isVencimento5Dias(): bool
    {
        return $this->tipo === self::TIPO_5_DIAS_ANTES;
    }
    
    public function isVencimento3Dias(): bool
    {
        return $this->tipo === self::TIPO_3_DIAS_ANTES;
    }
    
    public function isVenceHoje(): bool
    {
        return $this->tipo === self::TIPO_VENCE_HOJE;
    }
    
    public function isVencido(): bool
    {
        return $this->tipo === self::TIPO_VENCIDO;
    }
    
    public function isRecibo(): bool
    {
        return $this->tipo === self::TIPO_RECIBO;
    }
    
    public function isVencimento7Dias(): bool
    {
        return $this->tipo === self::TIPO_7_DIAS_ANTES;
    }
    
    public static function tiposValidos(): array
    {
        return [
            self::TIPO_5_DIAS_ANTES,
            self::TIPO_3_DIAS_ANTES,
            self::TIPO_VENCE_HOJE,
            self::TIPO_VENCIDO,
            self::TIPO_RECIBO,
            self::TIPO_PERSONALIZADO,
            self::TIPO_7_DIAS_ANTES,
        ];
    }
    
    public static function descricaoTipo(string $tipo): string
    {
        return match($tipo) {
            self::TIPO_5_DIAS_ANTES => '5 dias antes do vencimento',
            self::TIPO_3_DIAS_ANTES => '3 dias antes do vencimento',
            self::TIPO_VENCE_HOJE => 'No dia do vencimento',
            self::TIPO_VENCIDO => 'Parcela vencida (atraso)',
            self::TIPO_RECIBO => 'Recibo de pagamento',
            self::TIPO_PERSONALIZADO => 'Mensagem personalizada',
            self::TIPO_7_DIAS_ANTES => '7 dias antes do vencimento',
            default => 'Tipo desconhecido',
        };
    }
    
    // Garante que tipo é sempre string
    public function setTipoAttribute($value): void
    {
        $this->attributes['tipo'] = (string) $value;
    }
}

