<?php

namespace App\Http\Controllers;

use App\Models\Conexao;
use App\Services\WhatsAppService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

/**
 * Controller para gerenciar conexão WhatsApp
 * Migrado do sistema legado: /legado/master/whatsapp.php
 */
class WhatsAppController extends Controller
{
    /**
     * Página de gerenciamento do WhatsApp
     */
    public function index()
    {
        $user = Auth::user();
        
        // Busca ou cria registro de conexão
        $conexao = Conexao::firstOrCreate(
            ['id_usuario' => $user->id],
            ['conn' => 0, 'apikey' => '0', 'qrcode' => '']
        );

        // Verifica status da conexão se tiver API configurada
        $conectado = false;
        if ($user->whatsapp_api && $user->tokenapi) {
            $whatsapp = new WhatsAppService(
                $user->whatsapp_api,
                $user->whatsapp_token ?? $user->tokenapi,
                $user->tokenapi
            );
            
            $status = $whatsapp->getConnectionStatus();
            $conectado = $status['connected'] ?? false;
            
            // Atualiza status no banco
            if ($conectado != $conexao->conn) {
                $conexao->update(['conn' => $conectado ? 1 : 0]);
            }
        }

        return view('whatsapp.index', compact('conexao', 'conectado'));
    }

    /**
     * Gera QR Code para conectar WhatsApp
     */
    public function connect(Request $request)
    {
        $user = Auth::user();

        if (!$user->whatsapp_api || !$user->tokenapi) {
            return back()->with('error', 'Configure a API do WhatsApp nas configurações primeiro.');
        }

        try {
            $whatsapp = new WhatsAppService(
                $user->whatsapp_api,
                $user->whatsapp_token ?? $user->tokenapi,
                $user->tokenapi
            );

            $qrData = $whatsapp->generateQRCode();

            if (!$qrData || !$qrData['qrcode']) {
                return back()->with('erro', 'Não foi possível gerar o QR Code. Tente novamente.');
            }

            // Atualiza conexão
            $conexao = Conexao::firstOrCreate(
                ['id_usuario' => $user->id],
                ['conn' => 0, 'apikey' => '0']
            );

            $conexao->update([
                'qrcode' => $qrData['qrcode'],
                'conn' => 0,
                'apikey' => $user->tokenapi,
            ]);

            return back()->with('qrcode', $qrData['qrcode']);

        } catch (\Exception $e) {
            Log::error('Erro ao gerar QR Code WhatsApp', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return back()->with('erro', 'Erro ao conectar WhatsApp. Verifique suas configurações.');
        }
    }

    /**
     * Desconecta WhatsApp
     */
    public function disconnect(Request $request)
    {
        $user = Auth::user();

        if (!$user->whatsapp_api || !$user->tokenapi) {
            return back()->with('error', 'API do WhatsApp não configurada.');
        }

        try {
            $whatsapp = new WhatsAppService(
                $user->whatsapp_api,
                $user->whatsapp_token ?? $user->tokenapi,
                $user->tokenapi
            );

            $disconnected = $whatsapp->disconnect();

            // Atualiza conexão
            $conexao = Conexao::where('id_usuario', $user->id)->first();
            if ($conexao) {
                $conexao->update([
                    'qrcode' => '',
                    'conn' => 0,
                    'apikey' => '0',
                ]);
            }

            if ($disconnected) {
                return redirect()->route('whatsapp.index')->with('sucesso', 'Desconectado com sucesso!');
            } else {
                return back()->with('erro', 'Erro ao desconectar.');
            }

        } catch (\Exception $e) {
            Log::error('Erro ao desconectar WhatsApp', [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);

            return back()->with('erro', 'Erro ao desconectar WhatsApp.');
        }
    }

    /**
     * Verifica status da conexão (Ajax)
     */
    public function status(Request $request)
    {
        $user = Auth::user();

        if (!$user->whatsapp_api || !$user->tokenapi) {
            return response()->json(['connected' => false, 'message' => 'API não configurada']);
        }

        try {
            $whatsapp = new WhatsAppService(
                $user->whatsapp_api,
                $user->whatsapp_token ?? $user->tokenapi,
                $user->tokenapi
            );

            $status = $whatsapp->getConnectionStatus();

            return response()->json([
                'connected' => $status['connected'],
                'state' => $status['state']
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'connected' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
