<?php

namespace App\Http\Controllers\Pdv;

use App\Http\Controllers\Controller;
use App\Models\PdvProduto;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class EstoqueController extends Controller
{
    /**
     * Lista produtos com controle de estoque
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        $query = PdvProduto::where('idm', $user->id)
            ->with(['fornecedor', 'categoria']);

        // Filtros
        if ($request->has('estoque_baixo') && $request->estoque_baixo) {
            $query->estoqueBaixo();
        }

        if ($request->has('busca') && $request->busca) {
            $query->where(function($q) use ($request) {
                $q->where('nome', 'like', '%' . $request->busca . '%')
                  ->orWhere('codigo_barras', 'like', '%' . $request->busca . '%');
            });
        }

        $produtos = $query->orderBy('nome')->paginate(20);

        // Estatísticas
        $totalProdutos = PdvProduto::where('idm', $user->id)->count();
        $totalEstoque = PdvProduto::where('idm', $user->id)->sum('estoque');
        $produtosBaixo = PdvProduto::where('idm', $user->id)->estoqueBaixo()->count();
        
        // Valor total em estoque
        $valorTotal = PdvProduto::where('idm', $user->id)
            ->selectRaw('SUM(estoque * preco_venda) as total')
            ->first()
            ->total ?? 0;

        return view('pdv.estoque.index', compact('produtos', 'totalProdutos', 'totalEstoque', 'produtosBaixo', 'valorTotal'));
    }

    /**
     * Adiciona estoque
     */
    public function adicionar(Request $request, PdvProduto $produto)
    {
        $user = Auth::user();

        // Verificar se o produto pertence ao usuário
        if ($produto->idm !== $user->id) {
            abort(403);
        }

        $request->validate([
            'quantidade' => 'required|integer|min:1',
        ]);

        $produto->adicionarEstoque($request->quantidade);

        return redirect()->route('pdv.estoque.index')
            ->with('success', "Estoque adicionado com sucesso! Novo estoque: {$produto->fresh()->estoque}");
    }

    /**
     * Remove estoque
     */
    public function remover(Request $request, PdvProduto $produto)
    {
        $user = Auth::user();

        // Verificar se o produto pertence ao usuário
        if ($produto->idm !== $user->id) {
            abort(403);
        }

        $request->validate([
            'quantidade' => 'required|integer|min:1',
        ]);

        if ($produto->estoque < $request->quantidade) {
            return back()->withErrors(['error' => 'Estoque insuficiente!']);
        }

        $produto->removerEstoque($request->quantidade);

        return redirect()->route('pdv.estoque.index')
            ->with('success', "Estoque removido com sucesso! Novo estoque: {$produto->fresh()->estoque}");
    }

    /**
     * Ajuste manual de estoque
     */
    public function ajustar(Request $request, PdvProduto $produto)
    {
        $user = Auth::user();

        // Verificar se o produto pertence ao usuário
        if ($produto->idm !== $user->id) {
            abort(403);
        }

        $request->validate([
            'estoque' => 'required|integer|min:0',
        ]);

        $produto->update(['estoque' => $request->estoque]);

        return redirect()->route('pdv.estoque.index')
            ->with('success', "Estoque ajustado com sucesso! Novo estoque: {$produto->estoque}");
    }

    /**
     * Lista produtos com estoque baixo (API)
     */
    public function estoqueBaixo()
    {
        $user = Auth::user();

        $produtos = PdvProduto::where('idm', $user->id)
            ->estoqueBaixo()
            ->orderBy('estoque')
            ->get();

        return response()->json([
            'success' => true,
            'produtos' => $produtos->map(function($produto) {
                return [
                    'id' => $produto->id,
                    'nome' => $produto->nome,
                    'estoque' => $produto->estoque,
                    'estoque_minimo' => $produto->estoque_minimo,
                    'preco_venda' => $produto->preco_venda,
                ];
            })
        ]);
    }
}

