<?php

namespace App\Http\Controllers\Pdv;

use App\Http\Controllers\Controller;
use App\Models\PdvCaixa;
use App\Services\PdvService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CaixaController extends Controller
{
    protected $pdvService;

    public function __construct(PdvService $pdvService)
    {
        $this->pdvService = $pdvService;
    }

    /**
     * Lista todos os caixas
     */
    public function index()
    {
        $user = Auth::user();
        
        $caixas = PdvCaixa::where('idm', $user->id)
            ->orderBy('abertura', 'desc')
            ->paginate(20);

        return view('pdv.caixa.index', compact('caixas'));
    }

    /**
     * Abre um novo caixa
     */
    public function abrir(Request $request)
    {
        // Garantir que saldo_inicial tenha valor padrão
        $saldoInicial = $request->input('saldo_inicial', 0);
        if ($saldoInicial === '' || $saldoInicial === null) {
            $saldoInicial = 0;
        }

        $request->merge(['saldo_inicial' => $saldoInicial]);

        $request->validate([
            'saldo_inicial' => 'required|numeric|min:0',
            'atendente_nome' => 'nullable|string|max:255',
        ], [
            'saldo_inicial.required' => 'O saldo inicial é obrigatório.',
            'saldo_inicial.numeric' => 'O saldo inicial deve ser um número.',
            'saldo_inicial.min' => 'O saldo inicial não pode ser negativo.',
        ]);

        $user = Auth::user();

        if (!$user) {
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Usuário não autenticado!'
                ], 401);
            }
            return redirect()->route('login');
        }

        // Verificar se o usuário realmente existe no banco (sistema usa Carteira)
        $userExists = \App\Models\Carteira::find($user->id);
        if (!$userExists) {
            \Log::error('Usuário da sessão não existe no banco', [
                'session_user_id' => $user->id,
                'session_user_email' => $user->email ?? 'N/A',
            ]);
            
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Usuário da sessão inválido. Por favor, faça login novamente.'
                ], 401);
            }
            
            Auth::logout();
            return redirect()->route('login')->withErrors(['error' => 'Sessão inválida. Faça login novamente.']);
        }

        try {
            // Converter saldo_inicial para float
            $saldoInicial = (float) $request->saldo_inicial;
            
            \Log::info('Tentando abrir caixa', [
                'user_id' => $user->id,
                'user_email' => $user->email,
                'user_exists' => $userExists ? 'sim' : 'não',
                'saldo_inicial' => $saldoInicial,
            ]);
            
            $caixa = $this->pdvService->abrirCaixa(
                $user->id,
                $saldoInicial,
                $request->atendente_nome
            );

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Caixa aberto com sucesso!',
                    'caixa' => $caixa
                ]);
            }

            return redirect()->route('pdv.index')
                ->with('success', 'Caixa aberto com sucesso!');
        } catch (\Exception $e) {
            \Log::error('Erro ao abrir caixa', [
                'user_id' => Auth::id(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            $message = $e->getMessage();
            
            // Mensagens mais amigáveis
            if (strpos($message, 'Integrity constraint') !== false) {
                $message = 'Erro ao abrir caixa. Verifique se já existe um caixa aberto ou se há algum problema com os dados.';
            }

            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => $message,
                    'error' => $e->getMessage()
                ], 422);
            }

            return back()->withErrors(['error' => $message])->withInput();
        }
    }

    /**
     * Fecha um caixa
     */
    public function fechar(Request $request, PdvCaixa $caixa)
    {
        $user = Auth::user();

        if ($caixa->idm !== $user->id) {
            abort(403);
        }

        try {
            $this->pdvService->fecharCaixa($caixa);

            return redirect()->route('pdv.caixa.index')
                ->with('success', 'Caixa fechado com sucesso!');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => $e->getMessage()]);
        }
    }

    /**
     * Retorna o status do caixa atual (API)
     */
    public function status()
    {
        $user = Auth::user();

        $caixa = PdvCaixa::where('idm', $user->id)
            ->aberto()
            ->first();

        if ($caixa) {
            return response()->json([
                'aberto' => true,
                'caixa' => [
                    'id' => $caixa->id,
                    'saldo_inicial' => $caixa->saldo_inicial,
                    'atendente_nome' => $caixa->atendente_nome,
                    'abertura' => $caixa->abertura->format('d/m/Y H:i'),
                    'total_vendas' => $caixa->total_vendas,
                ],
            ]);
        }

        return response()->json([
            'aberto' => false,
            'caixa' => null,
        ]);
    }
}

