<?php

namespace App\Http\Controllers;

use App\Models\Carteira;
use App\Models\Cliente;
use App\Models\Financeiro1;
use App\Models\Financeiro2;
use App\Models\FinanceiroGlobal;
use App\Services\PaymentService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

/**
 * Controller para criação e gerenciamento de pagamentos
 * Migrado do sistema legado: /legado/master/create_payment.php e /legado/master/api/gerar_pix_link.php
 */
class PaymentController extends Controller
{
    /**
     * Gera pagamento para uma parcela
     */
    public function gerarPagamento(Request $request)
    {
        try {
            $user = Auth::user();
            $carteira = Carteira::find($user->Id);

            if (!$carteira) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Carteira não encontrada'
                ], 404);
            }

            $request->validate([
                'instancia' => 'required|exists:financeiro2,id',
                'type' => 'nullable|in:pix,boleto',
            ]);

            $parcela = Financeiro2::find($request->instancia);
            if (!$parcela) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Parcela não encontrada'
                ], 404);
            }

            // Verifica se já existe pagamento gerado
            $finGlobal = FinanceiroGlobal::where('instancia', $parcela->id)->first();

            if ($finGlobal && $finGlobal->status_payment === 'approved') {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Este pagamento já foi realizado'
                ], 400);
            }

            // Se já existe e está pendente, retorna os dados existentes
            if ($finGlobal && $finGlobal->copiacola && $finGlobal->imagemQrcode) {
                return response()->json([
                    'status' => 'success',
                    'mensagem' => 'PIX já existe',
                    'qrcode' => $finGlobal->imagemQrcode,
                    'copiacola' => $finGlobal->copiacola,
                    'payment_id' => $finGlobal->id_payment,
                    'ja_existia' => true
                ]);
            }

            // Busca cliente
            $cliente = Cliente::find($parcela->idc);
            if (!$cliente) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Cliente não encontrado'
                ], 404);
            }

            // Determina tipo de pagamento
            $type = $request->type ?? 'pix';
            $gateway = $carteira->pagamentos ?? $carteira->gatewayPayment ?? null;

            // Prepara dados para o service
            $dados = [
                'valor' => (float) $parcela->parcela,
                'vencimento' => $parcela->datapagamento,
                'name' => $cliente->nome,
                'nome' => $cliente->nome,
                'cpf' => $cliente->cpf,
                'email' => $cliente->email ?? '',
                'telefone' => $cliente->celular ?? '',
                'endereco' => $cliente->endereco ?? '',
                'numero' => $cliente->numero ?? '',
                'bairro' => $cliente->bairro ?? '',
                'cidade' => $cliente->cidade ?? '',
                'uf' => $cliente->estado ?? '',
                'cep' => $cliente->cep ?? '',
                'descricao' => 'Parcela - Cobrança',
                'order_id' => 'parcela_' . $parcela->id,
                'cliente_db' => $cliente,
                'external_reference' => $parcela->id,
            ];

            // Gera pagamento
            $paymentService = new PaymentService($carteira);
            $result = $paymentService->gerarPagamento($dados, $type);

            if (!$result || !$result['success']) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Erro ao gerar pagamento: ' . ($result['message'] ?? 'Erro desconhecido')
                ], 500);
            }

            // Cria ou atualiza financeiro_global
            if ($finGlobal) {
                $finGlobal->update([
                    'id_payment' => $result['id_payment'] ?? '',
                    'status_payment' => $result['status_payment'] ?? 'pending',
                    'copiacola' => $result['copiacola'] ?? '',
                    'imagemQrcode' => $result['imagemQrcode'] ?? '',
                    'codigobarra' => $result['codigobarra'] ?? '',
                    'type' => $type,
                ]);
            } else {
                FinanceiroGlobal::create([
                    'idc' => $cliente->id,
                    'idm' => $carteira->Id,
                    'instancia' => $parcela->id,
                    'valorparcela' => $result['valorparcela'] ?? $parcela->parcela,
                    'gatewayPayment' => $gateway,
                    'type' => $type,
                    'id_payment' => $result['id_payment'] ?? '',
                    'status_payment' => $result['status_payment'] ?? 'pending',
                    'copiacola' => $result['copiacola'] ?? '',
                    'imagemQrcode' => $result['imagemQrcode'] ?? '',
                    'codigobarra' => $result['codigobarra'] ?? '',
                ]);
            }

            return response()->json([
                'status' => 'success',
                'mensagem' => 'Pagamento gerado com sucesso',
                'qrcode' => $result['imagemQrcode'] ?? '',
                'copiacola' => $result['copiacola'] ?? '',
                'payment_id' => $result['id_payment'] ?? '',
                'ja_existia' => false
            ]);

        } catch (\Exception $e) {
            Log::error('Erro ao gerar pagamento', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'status' => 'error',
                'message' => 'Erro ao gerar pagamento: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Consulta status de um pagamento
     */
    public function consultarStatus(Request $request)
    {
        try {
            $user = Auth::user();
            $carteira = Carteira::find($user->Id);

            $request->validate([
                'payment_id' => 'required',
                'gateway' => 'required|in:1,2,3,4,5',
            ]);

            $paymentService = new PaymentService($carteira);
            $status = $paymentService->verificarStatus($request->payment_id, $request->gateway);

            return response()->json([
                'status' => 'success',
                'data' => $status
            ]);

        } catch (\Exception $e) {
            Log::error('Erro ao consultar status', [
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 'error',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}

