<?php

namespace App\Http\Controllers;

use App\Models\Carteira;
use App\Models\Cliente;
use App\Models\Financeiro1;
use App\Models\Financeiro2;
use App\Models\Financeiro3;
use App\Models\Conexao;
use App\Models\PdvCaixa;
use App\Models\Plano;
use App\Services\LicenseService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    protected LicenseService $licenseService;

    public function __construct(LicenseService $licenseService)
    {
        $this->licenseService = $licenseService;
    }

    public function index(Request $request)
    {
        $user = Auth::user();
        $mesParametro = $request->get('mes');
        $mes = $mesParametro ?? date('m');
        $ano = date('Y');
        $datam = $mes . '/' . $ano;

        // SOMA VALORES A RECEBER
        $valoresAReceber = Financeiro2::where('status', '1')
            ->where('datapagamento', 'like', '%' . $datam . '%')
            ->where('idm', $user->id)
            ->sum('parcela');

        // SOMA VALORES RECEBIDOS NO MÊS
        $valoresRecebidos = Financeiro2::where('status', '2')
            ->where('pagoem', 'like', '%' . $datam . '%')
            ->where('idm', $user->id)
            ->sum('parcela');

        // SOMA VALORES RECEBIDOS HOJE
        $hoje = date('d/m/Y');
        $valoresRecebidosHoje = Financeiro2::where('status', '2')
            ->where('pagoem', $hoje)
            ->where('idm', $user->id)
            ->sum('parcela');

        // COBRANÇAS ATIVAS
        $cobrancasAtivas = Financeiro1::where('status', '1')
            ->where('idm', $user->id)
            ->count();

        // PARCELAS ABERTAS
        $parcelasAbertas = Financeiro2::where('status', '1')
            ->where('datapagamento', 'like', '%' . $datam . '%')
            ->where('idm', $user->id)
            ->count();

        // PARCELAS PAGAS
        $parcelasPagas = Financeiro2::where('status', '2')
            ->where('pagoem', 'like', '%' . $datam . '%')
            ->where('idm', $user->id)
            ->count();

        // CLIENTES
        $totalClientes = Cliente::where('idm', $user->id)->count();

        // CONTAS A PAGAR
        $dataAtual = $ano . '-' . $mes;
        $valoresAPagar = Financeiro3::where('status', '1')
            ->where('datavencimento', 'like', '%' . $datam . '%')
            ->where('idm', $user->id)
            ->sum('valor');

        $valoresPagos = Financeiro3::where('status', '2')
            ->where('datapagamento', 'like', '%' . $dataAtual . '%')
            ->where('idm', $user->id)
            ->sum('valor');

        // VERIFICA CONEXÃO WHATSAPP
        $conexao = Conexao::where('id_usuario', $user->id)
            ->where('conn', '1')
            ->first();

        $whatsappConectado = false;
        if ($conexao) {
            // TODO: Verificar status real na API WhatsApp
            $whatsappConectado = true;
        }

        $meses = [
            '01' => 'Janeiro',
            '02' => 'Fevereiro',
            '03' => 'Março',
            '04' => 'Abril',
            '05' => 'Maio',
            '06' => 'Junho',
            '07' => 'Julho',
            '08' => 'Agosto',
            '09' => 'Setembro',
            '10' => 'Outubro',
            '11' => 'Novembro',
            '12' => 'Dezembro'
        ];

        // Verificador de versão (apenas para admin)
        $novaVersaoDisponivel = false;
        $versaoRemota = null;
        if ($user->tipo == 1) {
            $novaVersaoDisponivel = $this->licenseService->hasNewVersion();
            $versaoRemota = $this->licenseService->getRemoteVersion();
        }

        // STATUS DO CAIXA PDV
        $caixaAberto = PdvCaixa::where('idm', $user->id)
            ->where('status', 'aberto')
            ->orderBy('abertura', 'desc')
            ->first();
        
        $caixaStatus = [
            'aberto' => $caixaAberto ? true : false,
            'id' => $caixaAberto ? $caixaAberto->id : null,
            'atendente' => $caixaAberto ? $caixaAberto->atendente_nome : null,
            'data_abertura' => $caixaAberto ? $caixaAberto->abertura : null,
        ];

        // DADOS SAAS (apenas superadmin)
        $dadosSaas = null;
        if ($user->id == 1) {
            $totalUsuariosSaas = Carteira::count();
            $usuariosAtivos = Carteira::where('pagamentos', '2')->count();
            $totalPlanos = Plano::count();
            
            // Últimos 10 usuários cadastrados
            $ultimosUsuarios = Carteira::where('tipo', '!=', '1')
                ->where('created_at', '<=', now())
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get(['id', 'nome', 'login', 'created_at', 'tipo', 'assinatura']);
            
            $dadosSaas = [
                'total_usuarios' => $totalUsuariosSaas,
                'usuarios_ativos' => $usuariosAtivos,
                'total_planos' => $totalPlanos,
                'ultimos_usuarios' => $ultimosUsuarios,
            ];
        }

        // GRÁFICO DE RECEITA (últimos 7 dias)
        $receitaLabels = [];
        $receitaSeriesRecebido = [];
        $receitaSeriesPrevisto = [];
        $receitaTotalRecebido = 0;
        $receitaTotalPrevisto = 0;

        for ($i = 6; $i >= 0; $i--) {
            $dataDia = Carbon::today()->subDays($i);
            $dataFormatada = $dataDia->format('d/m');
            $dataSQL = $dataDia->format('Y-m-d');
            
            $receitaLabels[] = $dataFormatada;
            
            // Recebido
            $recebido = Financeiro2::where('status', '2')
                ->where('idm', $user->id)
                ->whereDate(DB::raw("COALESCE(STR_TO_DATE(pagoem, '%Y-%m-%d %H:%i:%s'), STR_TO_DATE(pagoem, '%Y-%m-%d'), STR_TO_DATE(pagoem, '%d/%m/%Y'))"), $dataSQL)
                ->sum('parcela');
            
            // Previsto
            $previsto = Financeiro2::where('status', '1')
                ->where('idm', $user->id)
                ->whereDate(DB::raw("COALESCE(STR_TO_DATE(datapagamento, '%Y-%m-%d'), STR_TO_DATE(datapagamento, '%d/%m/%Y'))"), $dataSQL)
                ->sum('parcela');
            
            $receitaSeriesRecebido[] = (float) $recebido;
            $receitaSeriesPrevisto[] = (float) $previsto;
            $receitaTotalRecebido += (float) $recebido;
            $receitaTotalPrevisto += (float) $previsto;
        }

        // DADOS DE LICENÇA
        $licenseInfo = $this->licenseService->validateLicense(request()->getHost());
        $licencaDiasRestantes = 0;
        $licencaDataExpiracao = null;
        if (isset($licenseInfo['expiration_date']) && $licenseInfo['expiration_date']) {
            try {
                $dataExp = Carbon::parse($licenseInfo['expiration_date']);
                $licencaDiasRestantes = now()->diffInDays($dataExp, false);
                $licencaDataExpiracao = $dataExp->format('d/m/Y');
            } catch (\Exception $e) {
                // Ignora erro de parsing
            }
        }

        return view('dashboard.index', compact(
            'valoresAReceber',
            'valoresRecebidos',
            'valoresRecebidosHoje',
            'cobrancasAtivas',
            'parcelasAbertas',
            'parcelasPagas',
            'totalClientes',
            'valoresAPagar',
            'valoresPagos',
            'whatsappConectado',
            'mes',
            'ano',
            'meses',
            'novaVersaoDisponivel',
            'versaoRemota',
            'caixaStatus',
            'dadosSaas',
            'receitaLabels',
            'receitaSeriesRecebido',
            'receitaSeriesPrevisto',
            'receitaTotalRecebido',
            'receitaTotalPrevisto',
            'licencaDiasRestantes',
            'licencaDataExpiracao',
            'mesParametro'
        ));
    }
}

