<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class ConfiguracaoController extends Controller
{
    /**
     * Exibe configurações
     */
    public function index()
    {
        $user = Auth::user();
        
        // Buscar configurações de email (apenas para super admin)
        $emailConfig = null;
        if ($user->id == 1) {
            try {
                if (DB::getSchemaBuilder()->hasTable('email_config')) {
                    $emailConfig = DB::table('email_config')->first();
                }
            } catch (\Exception $e) {
                // Tabela não existe
            }
        }
        
        // Buscar configurações de indicação (apenas para super admin)
        $indicacaoConfig = [];
        if ($user->id == 1) {
            try {
                if (DB::getSchemaBuilder()->hasTable('indicacoes_config')) {
                    $configs = DB::table('indicacoes_config')->get();
                    foreach ($configs as $config) {
                        $indicacaoConfig[$config->chave] = $config->valor;
                    }
                }
            } catch (\Exception $e) {
                // Tabela não existe
            }
        }
        
        // Total de indicações (apenas para super admin)
        $totalIndicacoes = 0;
        if ($user->id == 1) {
            try {
                if (DB::getSchemaBuilder()->hasTable('indicacoes')) {
                    $totalIndicacoes = DB::table('indicacoes')->count();
                }
            } catch (\Exception $e) {
                // Tabela não existe
            }
        }
        
        // URL base para webhooks
        $url = url('/');
        
        return view('configuracoes.index', compact('user', 'emailConfig', 'indicacaoConfig', 'totalIndicacoes', 'url'));
    }

    /**
     * Atualiza configurações gerais
     */
    public function update(Request $request)
    {
        $user = Auth::user();

        $data = [];
        
        // Dados da empresa
        if ($request->filled('nomecom')) {
            $data['nomecom'] = $request->nomecom;
        }
        if ($request->filled('cnpj')) {
            $data['cnpj'] = $request->cnpj;
        }
        if ($request->filled('enderecom')) {
            $data['enderecom'] = $request->enderecom;
        }
        if ($request->filled('contato')) {
            $data['contato'] = $request->contato;
        }
        if ($request->filled('senha')) {
            $data['senha'] = sha1($request->senha);
        }
        if ($request->filled('mensagem_ajuda')) {
            $data['mensagem_ajuda'] = $request->mensagem_ajuda;
        }
        
        // Modelo de cobrança
        if ($request->filled('tipopgmto')) {
            $data['pagamentos'] = $request->tipopgmto;
        }
        
        // Evitar cobrança fim de semana
        $data['nao_cobrar_fim_semana'] = $request->has('nao_cobrar_fim_semana') ? '1' : '0';
        
        // Tipo de disparo automático
        if ($request->filled('auto_cob_usar_link')) {
            $data['auto_cob_usar_link'] = $request->auto_cob_usar_link;
        }
        
        // Juros diários
        if ($request->filled('juros_diarios')) {
            $juros = str_replace(',', '.', $request->juros_diarios);
            $data['juros_diarios'] = floatval($juros);
        }
        
        // Background e Favicon (apenas texto para background, arquivo para favicon)
        if ($request->filled('background') && !$request->hasFile('background')) {
            $data['background'] = $request->background;
        }
        
        // Upload de favicon
        if ($request->hasFile('favicon')) {
            $file = $request->file('favicon');
            $filename = 'favicon_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('img'), $filename);
            $data['favicon'] = '/img/' . $filename;
        }
        
        // Upload de background (arquivo)
        if ($request->hasFile('background')) {
            $file = $request->file('background');
            $filename = 'background_' . time() . '.' . $file->getClientOriginalExtension();
            $file->move(public_path('img'), $filename);
            $data['background'] = '/img/' . $filename;
        }
        
        // Upload de logos (apenas superadmin)
        if ($user->tipo == 1) {
            if ($request->hasFile('logo_light')) {
                $file = $request->file('logo_light');
                $filename = 'logo_light_' . time() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('master/img'), $filename);
                // TODO: Salvar referência no banco
            }
            if ($request->hasFile('logo_dark')) {
                $file = $request->file('logo_dark');
                $filename = 'logo_dark_' . time() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('master/img'), $filename);
                // TODO: Salvar referência no banco
            }
        }
        
        // Campos Mercado Pago
        if ($request->filled('publicmp')) {
            $data['publicmp'] = $request->publicmp;
        }
        if ($request->filled('tokenmp')) {
            $data['tokenmp'] = $request->tokenmp;
        }
        
        // Campos QR Code e PIX
        if ($request->filled('msgqr')) {
            $data['msgqr'] = $request->msgqr;
        }
        if ($request->filled('msgpix')) {
            $data['msgpix'] = $request->msgpix;
        }
        
        // Campos Sem Gateway
        if ($request->filled('pix_manual_key')) {
            $data['pix_manual_key'] = $request->pix_manual_key;
        }
        
        // Campos PagHiper
        if ($request->filled('key_paghiper')) {
            $data['key_paghiper'] = $request->key_paghiper;
        }
        if ($request->filled('token_paghiper')) {
            $data['token_paghiper'] = $request->token_paghiper;
        }
        
        // Campos Asaas
        if ($request->filled('tokenasaas')) {
            $data['tokenasaas'] = $request->tokenasaas;
        }
        
        // Campos WhatsApp (apenas superadmin)
        if ($user->tipo == 1) {
            if ($request->filled('whatsapp_api')) {
                $data['whatsapp_api'] = $request->whatsapp_api;
            }
            if ($request->filled('whatsapp_token')) {
                $data['whatsapp_token'] = $request->whatsapp_token;
            }
            if ($request->filled('servidor_1_limite')) {
                $data['servidor_1_limite'] = $request->servidor_1_limite;
            }
            $data['servidor_1_ativo'] = $request->has('servidor_1_ativo') ? '1' : '0';
            
            if ($request->filled('whatsapp_api_2')) {
                $data['whatsapp_api_2'] = $request->whatsapp_api_2;
            }
            if ($request->filled('whatsapp_token_2')) {
                $data['whatsapp_token_2'] = $request->whatsapp_token_2;
            }
            if ($request->filled('servidor_2_limite')) {
                $data['servidor_2_limite'] = $request->servidor_2_limite;
            }
            $data['servidor_2_ativo'] = $request->has('servidor_2_ativo') ? '1' : '0';
        }
        
        // Atualizar dados do usuário
        $user->update($data);
        
        // Atualizar configurações de email (apenas superadmin)
        if ($user->id == 1) {
            try {
                if (DB::getSchemaBuilder()->hasTable('email_config')) {
                    $emailData = [
                        'email_verification_enabled' => $request->has('email_verification_enabled') ? 1 : 0,
                    ];
                    
                    if ($request->filled('smtp_host')) {
                        $emailData['smtp_host'] = $request->smtp_host;
                    }
                    if ($request->filled('smtp_port')) {
                        $emailData['smtp_port'] = $request->smtp_port;
                    }
                    if ($request->filled('smtp_encryption')) {
                        $emailData['smtp_encryption'] = $request->smtp_encryption;
                    }
                    if ($request->filled('smtp_username')) {
                        $emailData['smtp_username'] = $request->smtp_username;
                    }
                    if ($request->filled('smtp_password')) {
                        $emailData['smtp_password'] = $request->smtp_password;
                    }
                    if ($request->filled('from_email')) {
                        $emailData['from_email'] = $request->from_email;
                    }
                    if ($request->filled('from_name')) {
                        $emailData['from_name'] = $request->from_name;
                    }
                    
                    DB::table('email_config')->updateOrInsert(['id' => 1], $emailData);
                }
            } catch (\Exception $e) {
                // Tabela não existe ou erro
            }
            
            // Atualizar configurações de indicação (apenas superadmin)
            try {
                if (DB::getSchemaBuilder()->hasTable('indicacoes_config')) {
                    if ($request->filled('pontos_por_indicacao')) {
                        DB::table('indicacoes_config')->updateOrInsert(
                            ['chave' => 'pontos_por_indicacao'],
                            ['valor' => $request->pontos_por_indicacao]
                        );
                    }
                    if ($request->filled('minimo_resgate')) {
                        DB::table('indicacoes_config')->updateOrInsert(
                            ['chave' => 'minimo_resgate'],
                            ['valor' => $request->minimo_resgate]
                        );
                    }
                    if ($request->filled('expiracao_pontos_meses')) {
                        DB::table('indicacoes_config')->updateOrInsert(
                            ['chave' => 'expiracao_pontos_meses'],
                            ['valor' => $request->expiracao_pontos_meses]
                        );
                    }
                    DB::table('indicacoes_config')->updateOrInsert(
                        ['chave' => 'sistema_ativo'],
                        ['valor' => $request->has('sistema_indicacao_ativo') ? '1' : '0']
                    );
                }
            } catch (\Exception $e) {
                // Tabela não existe ou erro
            }
        }

        if ($request->expectsJson() || $request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Configurações atualizadas com sucesso!'
            ]);
        }

        return back()->with('success', 'Configurações atualizadas com sucesso!');
    }

    /**
     * Atualiza tokens de pagamento
     */
    public function updatePagamentos(Request $request)
    {
        $user = Auth::user();

        $data = [];

        // Mercado Pago
        if ($request->filled('tokenmp')) {
            $data['tokenmp'] = $request->tokenmp;
        }
        if ($request->filled('publicmp')) {
            $data['publicmp'] = $request->publicmp;
        }

        // Asaas
        if ($request->filled('tokenasaas')) {
            $data['tokenasaas'] = $request->tokenasaas;
        }
        if ($request->filled('client_id_asaas')) {
            $data['client_id_asaas'] = $request->client_id_asaas;
        }

        // Gerencianet/Efí
        if ($request->filled('gerecianet_client')) {
            $data['gerecianet_client'] = $request->gerecianet_client;
        }
        if ($request->filled('gerecianet_secret')) {
            $data['gerecianet_secret'] = $request->gerecianet_secret;
        }
        if ($request->filled('chave_pix')) {
            $data['chave_pix'] = $request->chave_pix;
        }

        // PagHiper
        if ($request->filled('key_paghiper')) {
            $data['key_paghiper'] = $request->key_paghiper;
        }
        if ($request->filled('token_paghiper')) {
            $data['token_paghiper'] = $request->token_paghiper;
        }

        // Gateway padrão
        if ($request->filled('pagamentos')) {
            $data['pagamentos'] = $request->pagamentos;
        }

        $user->update($data);

        return back()->with('success', 'Configurações atualizadas com sucesso!');
    }

    /**
     * Atualiza configurações WhatsApp
     */
    public function updateWhatsApp(Request $request)
    {
        $user = Auth::user();

        $data = $request->validate([
            'whatsapp_api' => 'nullable|url',
            'whatsapp_token' => 'nullable|max:255',
            'msg' => 'nullable|in:1,2',
            'msgqr' => 'nullable|in:1,2',
            'msgpix' => 'nullable|in:1,2',
        ]);

        $user->update($data);

        return back()->with('success', 'Configurações atualizadas com sucesso!');
    }

    /**
     * Perfil do usuário
     */
    public function perfil()
    {
        $user = Auth::user();
        return view('configuracoes.perfil', compact('user'));
    }

    /**
     * Atualiza perfil
     */
    public function updatePerfil(Request $request)
    {
        $user = Auth::user();

        $data = $request->validate([
            'nome' => 'required|max:120',
            'celular' => 'nullable|max:255',
            'email' => 'nullable|email|max:120',
            'cpf' => 'nullable|max:25',
            'nascimento' => 'nullable|max:15',
            'cep' => 'nullable|max:25',
            'endereco' => 'nullable|max:120',
            'numero' => 'nullable|max:9',
            'bairro' => 'nullable|max:60',
            'complemento' => 'nullable|max:60',
            'cidade' => 'nullable|max:30',
            'uf' => 'nullable|max:5',
        ]);

        // Atualizar senha se fornecida
        if ($request->filled('senha_nova')) {
            $request->validate([
                'senha_atual' => 'required',
                'senha_nova' => 'required|min:6|confirmed',
            ]);

            // Verifica senha atual
            if (sha1($request->senha_atual) !== $user->senha) {
                return back()->with('error', 'Senha atual incorreta.');
            }

            $data['senha'] = sha1($request->senha_nova);
        }

        $user->update($data);

        return back()->with('success', 'Configurações atualizadas com sucesso!');
    }

    /**
     * Alterna tema (dark/light)
     */
    public function toggleTheme(Request $request)
    {
        $user = Auth::user();
        
        $newTheme = $user->theme === 'dark' ? 'light' : 'dark';
        
        $user->update(['theme' => $newTheme]);

        return response()->json([
            'success' => true,
            'theme' => $newTheme
        ]);
    }

    /**
     * Upload de background personalizado
     */
    public function uploadBackground(Request $request)
    {
        $user = Auth::user();

        if ($user->tipo != 1) {
            return back()->with('error', 'Apenas administradores podem alterar o background.');
        }

        $request->validate([
            'background' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        try {
            if ($request->hasFile('background')) {
                $file = $request->file('background');
                $filename = 'background_' . time() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('img'), $filename);

                $user->update(['background' => '/img/' . $filename]);

                return back()->with('success', 'Background atualizado com sucesso!');
            }

            return back()->with('error', 'Nenhum arquivo foi enviado.');

        } catch (\Exception $e) {
            return back()->with('error', 'Erro ao fazer upload: ' . $e->getMessage());
        }
    }

    /**
     * Upload de favicon personalizado
     */
    public function uploadFavicon(Request $request)
    {
        $user = Auth::user();

        if ($user->tipo != 1) {
            return back()->with('error', 'Apenas administradores podem alterar o favicon.');
        }

        $request->validate([
            'favicon' => 'required|mimes:ico,png|max:512'
        ]);

        try {
            if ($request->hasFile('favicon')) {
                $file = $request->file('favicon');
                $filename = 'favicon_' . time() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('img'), $filename);

                $user->update(['favicon' => '/img/' . $filename]);

                return back()->with('success', 'Favicon atualizado com sucesso!');
            }

            return back()->with('error', 'Nenhum arquivo foi enviado.');

        } catch (\Exception $e) {
            return back()->with('error', 'Erro ao fazer upload: ' . $e->getMessage());
        }
    }

    /**
     * Testar conexão SMTP
     */
    public function testEmail(Request $request)
    {
        try {
            $user = Auth::user();
            
            // Verificar se é super admin
            if ($user->tipo != 1) {
                return response()->json([
                    'success' => false,
                    'message' => 'Apenas administradores podem testar a conexão SMTP.'
                ], 403);
            }

            $email = $request->input('email', $user->login);
            
            if (empty($email)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Email não informado.'
                ], 400);
            }

            // Buscar configurações de email
            $emailConfig = DB::table('email_config')->where('id', 1)->first();
            
            if (!$emailConfig || !$emailConfig->smtp_host) {
                return response()->json([
                    'success' => false,
                    'message' => 'Configurações de email não encontradas.'
                ], 400);
            }

            // Configurar mailer
            config([
                'mail.mailers.smtp.host' => $emailConfig->smtp_host,
                'mail.mailers.smtp.port' => $emailConfig->smtp_port,
                'mail.mailers.smtp.encryption' => $emailConfig->smtp_encryption,
                'mail.mailers.smtp.username' => $emailConfig->smtp_username,
                'mail.mailers.smtp.password' => $emailConfig->smtp_password,
                'mail.from.address' => $emailConfig->from_email,
                'mail.from.name' => $emailConfig->from_name ?? config('app.name'),
            ]);

            // Enviar email de teste
            \Mail::raw('Este é um email de teste do sistema. Se você recebeu esta mensagem, a configuração SMTP está funcionando corretamente.', function ($message) use ($email, $emailConfig) {
                $message->to($email)
                        ->subject('Teste de Conexão SMTP - ' . config('app.name'));
            });

            return response()->json([
                'success' => true,
                'message' => 'Email de teste enviado com sucesso para ' . $email . '. Verifique sua caixa de entrada.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao enviar email de teste: ' . $e->getMessage()
            ], 500);
        }
    }
}

