<?php

namespace App\Http\Controllers;

use App\Models\Funcionario;
use App\Models\PdvVenda;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ComissaoController extends Controller
{
    /**
     * Relatório geral de comissões
     */
    public function index(Request $request)
    {
        $user = Auth::user();

        // Filtros de data
        $periodo = $request->get('periodo', 'mes_atual');
        $dataInicio = $this->getDataInicio($periodo, $request);
        $dataFim = $this->getDataFim($periodo, $request);

        // Filtro de funcionário
        $funcionarioId = $request->get('funcionario_id', 'todos');

        // Estatísticas gerais
        $query = PdvVenda::where('idm', $user->id)
            ->where('comissao', '>', 0)
            ->whereBetween(DB::raw('DATE(data_venda)'), [$dataInicio, $dataFim]);

        if ($funcionarioId !== 'todos') {
            $query->where('funcionario_id', $funcionarioId);
        }

        $totalComissoes = $query->sum('comissao');
        $totalVendas = $query->count();
        $mediaComissao = $totalVendas > 0 ? $totalComissoes / $totalVendas : 0;

        // Total de funcionários com comissão
        $totalFuncionarios = $query->distinct('funcionario_id')->count('funcionario_id');

        // Top 3 funcionários por comissão
        $topComissao = PdvVenda::where('idm', $user->id)
            ->where('comissao', '>', 0)
            ->whereBetween(DB::raw('DATE(data_venda)'), [$dataInicio, $dataFim])
            ->when($funcionarioId !== 'todos', function($q) use ($funcionarioId) {
                $q->where('funcionario_id', $funcionarioId);
            })
            ->select('funcionario_id', 
                     DB::raw('COUNT(*) as total_vendas'), 
                     DB::raw('SUM(total) as total_faturado'), 
                     DB::raw('SUM(comissao) as total_comissao'))
            ->groupBy('funcionario_id')
            ->orderBy('total_comissao', 'desc')
            ->limit(3)
            ->get()
            ->map(function($item) {
                $item->funcionario = Funcionario::find($item->funcionario_id);
                return $item;
            });

        // Top 3 funcionários por quantidade de vendas
        $topVendas = PdvVenda::where('idm', $user->id)
            ->where('comissao', '>', 0)
            ->whereBetween(DB::raw('DATE(data_venda)'), [$dataInicio, $dataFim])
            ->when($funcionarioId !== 'todos', function($q) use ($funcionarioId) {
                $q->where('funcionario_id', $funcionarioId);
            })
            ->select('funcionario_id', 
                     DB::raw('COUNT(*) as total_vendas'), 
                     DB::raw('SUM(total) as total_faturado'), 
                     DB::raw('SUM(comissao) as total_comissao'))
            ->groupBy('funcionario_id')
            ->orderBy('total_vendas', 'desc')
            ->limit(3)
            ->get()
            ->map(function($item) {
                $item->funcionario = Funcionario::find($item->funcionario_id);
                return $item;
            });

        // Lista completa de funcionários com comissões
        $funcionariosComissao = PdvVenda::where('idm', $user->id)
            ->where('comissao', '>', 0)
            ->whereBetween(DB::raw('DATE(data_venda)'), [$dataInicio, $dataFim])
            ->when($funcionarioId !== 'todos', function($q) use ($funcionarioId) {
                $q->where('funcionario_id', $funcionarioId);
            })
            ->select('funcionario_id', 
                     DB::raw('COUNT(*) as total_vendas'), 
                     DB::raw('SUM(total) as total_faturado'), 
                     DB::raw('SUM(comissao) as total_comissao'),
                     DB::raw('AVG(comissao) as media_comissao'))
            ->groupBy('funcionario_id')
            ->orderBy('total_comissao', 'desc')
            ->get()
            ->map(function($item) {
                $item->funcionario = Funcionario::find($item->funcionario_id);
                return $item;
            });

        // Lista de funcionários para filtro
        $funcionarios = Funcionario::where('id_saas', $user->id)
            ->orderBy('nome')
            ->get();

        return view('comissoes.index', compact(
            'totalComissoes', 
            'totalVendas', 
            'mediaComissao', 
            'totalFuncionarios',
            'topComissao',
            'topVendas',
            'funcionariosComissao',
            'funcionarios',
            'periodo',
            'dataInicio',
            'dataFim',
            'funcionarioId'
        ));
    }

    /**
     * Relatório de comissões por funcionário
     */
    public function funcionario(Request $request, Funcionario $funcionario)
    {
        $user = Auth::user();

        if ($funcionario->id_saas !== $user->id) {
            abort(403);
        }

        // Filtros de data
        $periodo = $request->get('periodo', 'mes_atual');
        $dataInicio = $this->getDataInicio($periodo, $request);
        $dataFim = $this->getDataFim($periodo, $request);

        // Estatísticas do funcionário
        $query = PdvVenda::where('funcionario_id', $funcionario->id)
            ->where('idm', $user->id)
            ->whereBetween(DB::raw('DATE(data_venda)'), [$dataInicio, $dataFim]);

        $totalVendas = $query->count();
        $totalFaturado = $query->sum('total');
        $totalComissao = $query->where('comissao', '>', 0)->sum('comissao');
        $mediaComissao = $totalVendas > 0 ? $totalComissao / $totalVendas : 0;

        // Vendas do funcionário
        $vendas = PdvVenda::where('funcionario_id', $funcionario->id)
            ->where('idm', $user->id)
            ->whereBetween(DB::raw('DATE(data_venda)'), [$dataInicio, $dataFim])
            ->with(['cliente', 'caixa'])
            ->orderBy('data_venda', 'desc')
            ->paginate(20);

        // Comissões por mês (últimos 6 meses)
        $comissoesPorMes = PdvVenda::where('funcionario_id', $funcionario->id)
            ->where('idm', $user->id)
            ->where('comissao', '>', 0)
            ->where('data_venda', '>=', Carbon::now()->subMonths(6))
            ->select(
                DB::raw('YEAR(data_venda) as ano'),
                DB::raw('MONTH(data_venda) as mes'),
                DB::raw('SUM(comissao) as total_comissao'),
                DB::raw('COUNT(*) as total_vendas')
            )
            ->groupBy('ano', 'mes')
            ->orderBy('ano', 'desc')
            ->orderBy('mes', 'desc')
            ->get();

        return view('comissoes.funcionario', compact(
            'funcionario',
            'totalVendas',
            'totalFaturado',
            'totalComissao',
            'mediaComissao',
            'vendas',
            'comissoesPorMes',
            'periodo',
            'dataInicio',
            'dataFim'
        ));
    }

    /**
     * Retorna data de início baseada no período
     */
    private function getDataInicio(string $periodo, Request $request): string
    {
        $carbon = Carbon::now();
        
        switch ($periodo) {
            case 'hoje':
                return $carbon->format('Y-m-d');
            case 'ontem':
                return $carbon->copy()->subDay()->format('Y-m-d');
            case 'ultima_semana':
                return $carbon->copy()->subDays(7)->format('Y-m-d');
            case 'mes_atual':
                return $carbon->copy()->startOfMonth()->format('Y-m-d');
            case 'mes_passado':
                return $carbon->copy()->subMonth()->startOfMonth()->format('Y-m-d');
            case 'ultimos_30_dias':
                return $carbon->copy()->subDays(30)->format('Y-m-d');
            case 'ultimos_90_dias':
                return $carbon->copy()->subDays(90)->format('Y-m-d');
            case 'ano_atual':
                return $carbon->copy()->startOfYear()->format('Y-m-d');
            case 'personalizado':
                return $request->get('data_inicio', $carbon->copy()->startOfMonth()->format('Y-m-d'));
            default:
                return $carbon->copy()->startOfMonth()->format('Y-m-d');
        }
    }

    /**
     * Retorna data de fim baseada no período
     */
    private function getDataFim(string $periodo, Request $request): string
    {
        $carbon = Carbon::now();
        
        switch ($periodo) {
            case 'hoje':
            case 'ontem':
                return $this->getDataInicio($periodo, $request);
            case 'ultima_semana':
            case 'ultimos_30_dias':
            case 'ultimos_90_dias':
                return $carbon->format('Y-m-d');
            case 'mes_atual':
                return $carbon->copy()->endOfMonth()->format('Y-m-d');
            case 'mes_passado':
                return $carbon->copy()->subMonth()->endOfMonth()->format('Y-m-d');
            case 'ano_atual':
                return $carbon->copy()->endOfYear()->format('Y-m-d');
            case 'personalizado':
                return $request->get('data_fim', $carbon->format('Y-m-d'));
            default:
                return $carbon->copy()->endOfMonth()->format('Y-m-d');
        }
    }
}

