<?php

namespace App\Http\Controllers;

use App\Models\Cliente;
use App\Models\Categoria;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ClienteController extends Controller
{
    /**
     * Lista todos os clientes
     */
    public function index()
    {
        $user = Auth::user();
        
        $clientes = Cliente::where('idm', $user->id)
            ->with('categoria')
            ->orderBy('nome', 'asc')
            ->get();

        $totalCadastrados = $clientes->count();
        
        // Limite do plano
        $limite = $user->cadastros_lim ?? 0;
        $limiteDisponivel = $limite > 0 ? ($limite - $totalCadastrados) : 'Ilimitado';
        
        // Nome do plano
        $planoNome = 'Grátis';
        if ($user->plano_id) {
            $plano = \App\Models\Plano::find($user->plano_id);
            if ($plano) {
                $planoNome = $plano->nome;
            }
        }
        
        // Verificar se pode criar cliente
        $isSuperAdmin = ($user->tipo == 1);
        $hasLimitAvailable = (!$limite || $totalCadastrados < $limite);
        $canCreateClient = $isSuperAdmin || $hasLimitAvailable;
        
        // Categorias para filtro
        $categorias = Categoria::where('idu', $user->id)
            ->orderBy('nome', 'asc')
            ->get();

        return view('clientes.index', compact('clientes', 'totalCadastrados', 'limiteDisponivel', 'planoNome', 'limite', 'canCreateClient', 'categorias'));
    }

    /**
     * Exibe formulário de criar cliente
     */
    public function create()
    {
        $user = Auth::user();
        
        $categorias = Categoria::where('idu', $user->id)
            ->orderBy('nome', 'asc')
            ->get();

        return view('clientes.create', compact('categorias'));
    }

    /**
     * Salva novo cliente
     */
    public function store(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'nome' => 'required|max:120',
            'celular' => 'required',
            'email' => 'nullable|email|max:120',
            'cpf_cnpj' => 'nullable|max:25',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        Cliente::create([
            'idm' => $user->id,
            'idc' => 1, // Categoria padrão
            'nome' => $request->nome,
            'celular' => $request->celular,
            'email' => $request->email,
            'cpf' => $request->cpf_cnpj,
            'cep' => $request->cep,
            'endereco' => $request->endereco,
            'numero' => $request->numero,
            'bairro' => $request->bairro,
            'cidade' => $request->cidade,
            'estado' => $request->estado,
            'observacoes' => $request->observacoes,
            'nascimento' => null,
            'id_asaas' => null,
        ]);

        return redirect()->route('clientes.index')->with('success', 'Cliente cadastrado com sucesso!');
    }

    /**
     * Exibe detalhes do cliente
     */
    public function show($id)
    {
        $user = Auth::user();
        
        $cliente = Cliente::where('id', $id)
            ->where('idm', $user->id)
            ->with(['categoria', 'financeiro1', 'financeiro2'])
            ->firstOrFail();

        return view('clientes.show', compact('cliente'));
    }

    /**
     * Exibe formulário de edição
     */
    public function edit($id)
    {
        $user = Auth::user();
        
        $cliente = Cliente::where('id', $id)
            ->where('idm', $user->id)
            ->firstOrFail();

        $categorias = Categoria::where('idu', $user->id)
            ->orderBy('nome', 'asc')
            ->get();

        return view('clientes.edit', compact('cliente', 'categorias'));
    }

    /**
     * Atualiza cliente
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();

        $cliente = Cliente::where('id', $id)
            ->where('idm', $user->id)
            ->firstOrFail();

        $validator = Validator::make($request->all(), [
            'nome' => 'required|max:120',
            'celular' => 'required',
            'email' => 'nullable|email|max:120',
            'cpf_cnpj' => 'nullable|max:25',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $cliente->update([
            'nome' => $request->nome,
            'celular' => $request->celular,
            'email' => $request->email,
            'cpf' => $request->cpf_cnpj,
            'cep' => $request->cep,
            'endereco' => $request->endereco,
            'numero' => $request->numero,
            'bairro' => $request->bairro,
            'cidade' => $request->cidade,
            'estado' => $request->estado,
            'observacoes' => $request->observacoes,
        ]);

        return redirect()->route('clientes.index')->with('success', 'Cliente atualizado com sucesso!');
    }

    /**
     * Deleta cliente
     */
    public function destroy($id)
    {
        $user = Auth::user();

        $cliente = Cliente::where('id', $id)
            ->where('idm', $user->id)
            ->firstOrFail();

        $cliente->delete();

        return redirect()->route('clientes.index')->with('success', 'Cliente excluído com sucesso!');
    }
}

