<?php

namespace App\Console\Commands;

use App\Models\Financeiro2;
use App\Models\FinanceiroGlobal;
use App\Models\User;
use App\Models\MercadoPago;
use App\Services\GerencianetService;
use App\Services\PaghiperService;
use App\Services\AsaasService;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

/**
 * Command para verificar status de pagamentos nos gateways
 * Migrado do sistema legado: /legado/crons/conf_pgmto.php
 */
class VerificarPagamentosCommand extends Command
{
    protected $signature = 'pagamentos:verificar {--gateway= : Verificar apenas um gateway específico (1-5)}';
    protected $description = 'Verifica status de pagamentos em todos os gateways e atualiza parcelas pagas';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Iniciando verificação de pagamentos...');

        try {
            $gatewayFiltro = $this->option('gateway');
            
            $totalVerificados = 0;
            $totalAtualizados = 0;

            // Busca parcelas pendentes
            $parcelas = Financeiro2::where(function($query) {
                $query->where('pagoem', 'n')
                      ->orWhere('status', '1');
            })->get();

            $this->info("Encontradas {$parcelas->count()} parcelas pendentes.");

            foreach ($parcelas as $parcela) {
                $totalVerificados++;

                // Filtra por gateway se especificado
                if ($gatewayFiltro && $parcela->gatewayPayment != $gatewayFiltro) {
                    continue;
                }

                // Busca dados do master
                $master = User::find($parcela->idm);
                if (!$master) {
                    continue;
                }

                // Verifica pagamento baseado no gateway
                $pago = false;
                $dataPagamento = null;

                switch ($parcela->gatewayPayment) {
                    case 1:
                        // Mercado Pago
                        $resultado = $this->verificarMercadoPago($parcela, $master);
                        $pago = $resultado['pago'];
                        $dataPagamento = $resultado['data'];
                        break;

                    case 3:
                        // Gerencianet
                        $resultado = $this->verificarGerencianet($parcela, $master);
                        $pago = $resultado['pago'];
                        $dataPagamento = $resultado['data'];
                        break;

                    case 4:
                        // Paghiper
                        $resultado = $this->verificarPaghiper($parcela, $master);
                        $pago = $resultado['pago'];
                        $dataPagamento = $resultado['data'];
                        break;

                    case 5:
                        // Asaas
                        $resultado = $this->verificarAsaas($parcela, $master);
                        $pago = $resultado['pago'];
                        $dataPagamento = $resultado['data'];
                        break;

                    case 2:
                    default:
                        // Sem gateway - pular
                        continue 2;
                }

                // Se foi pago, atualiza a parcela
                if ($pago) {
                    $this->darBaixa($parcela, $dataPagamento);
                    $totalAtualizados++;
                    $this->info("Parcela #{$parcela->id} marcada como paga.");
                }
            }

            $this->info("Verificação concluída: {$totalVerificados} parcelas verificadas, {$totalAtualizados} atualizadas.");
            
            return Command::SUCCESS;

        } catch (\Exception $e) {
            Log::error('Erro ao verificar pagamentos', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            $this->error('Erro: ' . $e->getMessage());
            return Command::FAILURE;
        }
    }

    /**
     * Verifica pagamento no Mercado Pago
     */
    private function verificarMercadoPago(Financeiro2 $parcela, User $master): array
    {
        try {
            $mp = MercadoPago::where('instancia', $parcela->id)->first();
            if (!$mp || !$mp->payment_id) {
                return ['pago' => false, 'data' => null];
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $master->tokenmp
            ])->get("https://api.mercadopago.com/v1/payments/{$mp->payment_id}");

            if ($response->successful()) {
                $data = $response->json();
                $status = $data['status'] ?? '';
                
                if (in_array($status, ['approved', 'in_process'])) {
                    return [
                        'pago' => true,
                        'data' => isset($data['date_approved']) 
                            ? Carbon::parse($data['date_approved'])->format('d/m/Y')
                            : now()->format('d/m/Y')
                    ];
                }
            }

            return ['pago' => false, 'data' => null];

        } catch (\Exception $e) {
            Log::error('Erro ao verificar Mercado Pago', [
                'parcela_id' => $parcela->id,
                'error' => $e->getMessage()
            ]);
            return ['pago' => false, 'data' => null];
        }
    }

    /**
     * Verifica pagamento na Gerencianet
     */
    private function verificarGerencianet(Financeiro2 $parcela, User $master): array
    {
        try {
            if (!$master->gerecianet_client || !$master->gerecianet_secret) {
                return ['pago' => false, 'data' => null];
            }

            $finGlobal = FinanceiroGlobal::where('instancia', $parcela->id)->first();
            if (!$finGlobal || !$finGlobal->id_payment) {
                return ['pago' => false, 'data' => null];
            }

            $service = new GerencianetService(
                $master->gerecianet_client,
                $master->gerecianet_secret,
                $master->chave_pix ?? '',
                $master->certificado_pem
            );

            $status = $service->consultarStatusPix($finGlobal->id_payment);
            
            if ($status && $status['pago']) {
                return [
                    'pago' => true,
                    'data' => now()->format('d/m/Y')
                ];
            }

            return ['pago' => false, 'data' => null];

        } catch (\Exception $e) {
            Log::error('Erro ao verificar Gerencianet', [
                'parcela_id' => $parcela->id,
                'error' => $e->getMessage()
            ]);
            return ['pago' => false, 'data' => null];
        }
    }

    /**
     * Verifica pagamento no Paghiper
     */
    private function verificarPaghiper(Financeiro2 $parcela, User $master): array
    {
        try {
            if (!$master->key_paghiper || !$master->token_paghiper) {
                return ['pago' => false, 'data' => null];
            }

            $finGlobal = FinanceiroGlobal::where('instancia', $parcela->id)->first();
            if (!$finGlobal || !$finGlobal->id_payment) {
                return ['pago' => false, 'data' => null];
            }

            $service = new PaghiperService(
                $master->key_paghiper,
                $master->token_paghiper
            );

            $status = $service->consultarStatus($finGlobal->id_payment);
            
            if ($status && $status['pago']) {
                return [
                    'pago' => true,
                    'data' => now()->format('d/m/Y')
                ];
            }

            return ['pago' => false, 'data' => null];

        } catch (\Exception $e) {
            Log::error('Erro ao verificar Paghiper', [
                'parcela_id' => $parcela->id,
                'error' => $e->getMessage()
            ]);
            return ['pago' => false, 'data' => null];
        }
    }

    /**
     * Verifica pagamento no Asaas
     */
    private function verificarAsaas(Financeiro2 $parcela, User $master): array
    {
        try {
            if (!$master->client_id_asaas) {
                return ['pago' => false, 'data' => null];
            }

            $finGlobal = FinanceiroGlobal::where('instancia', $parcela->id)->first();
            if (!$finGlobal || !$finGlobal->id_payment) {
                return ['pago' => false, 'data' => null];
            }

            $service = new AsaasService($master->client_id_asaas);

            $status = $service->getPaymentStatus($finGlobal->id_payment);
            
            if ($status && $status['pago']) {
                return [
                    'pago' => true,
                    'data' => isset($status['paymentDate']) 
                        ? Carbon::parse($status['paymentDate'])->format('d/m/Y')
                        : now()->format('d/m/Y')
                ];
            }

            return ['pago' => false, 'data' => null];

        } catch (\Exception $e) {
            Log::error('Erro ao verificar Asaas', [
                'parcela_id' => $parcela->id,
                'error' => $e->getMessage()
            ]);
            return ['pago' => false, 'data' => null];
        }
    }

    /**
     * Dá baixa na parcela
     */
    private function darBaixa(Financeiro2 $parcela, ?string $dataPagamento): void
    {
        try {
            $parcela->update([
                'status' => '2',
                'pagoem' => $dataPagamento ?? now()->format('d/m/Y')
            ]);

            Log::info('Parcela atualizada para paga', [
                'parcela_id' => $parcela->id,
                'data_pagamento' => $dataPagamento
            ]);

        } catch (\Exception $e) {
            Log::error('Erro ao dar baixa na parcela', [
                'parcela_id' => $parcela->id,
                'error' => $e->getMessage()
            ]);
        }
    }
}
